unit RTLPatching;

// Brian Long - Embarcadero MVP
// Email - brian@blong.com
// Web  - http://blong.com
// Blog - http://blog.blong.com

{$IFDEF Ver100} { Delphi 3.0x }
  {$DEFINE Delphi3}
{$ENDIF}
{$IFDEF Ver110} { C++ Builder 3.0x }
  {$DEFINE Delphi3}
{$ENDIF}
{$IFDEF Ver120} { Delphi 4.0x }
  {$DEFINE Delphi4}
{$ENDIF}
{$IFDEF Ver125} { C++Builder 4.0x }
  {$DEFINE Delphi4}
{$ENDIF}
{$IFDEF ConditionalExpressions}
  {$DEFINE Delphi6AndAbove}
{$ENDIF}

{$IFDEF Delphi3}
  'This unit requires Delphi 4 or later'
{$ENDIF}

//Define either of these two symbols to decide what to reimplement
{$DEFINE HookAssertions}
{$DEFINE HookAutomation}

interface

var
{$IFDEF HookAutomation}
  Msg: Variant;
{$ENDIF}
  ShowPIDAndTIDInfo: Boolean = False;

implementation

uses
{$IFDEF UNICODE}
  AnsiStrings,
{$ENDIF}
{$IFDEF Delphi6AndAbove}
  Variants,
{$ENDIF}
  SysUtils, Windows, DebugSupport;

{$IFNDEF Delphi6AndAbove}
type
  TVarType = Word;
{$ENDIF}

{$REGION 'Automation hooking Take 1'}
procedure VarDispHandler1(Result: Pointer; const Instance: Variant;
  CallDesc: Pointer; Params: Pointer); cdecl;
begin
  DebugMsg(PChar(Params{$ifdef Delphi6AndAbove}^{$endif}))
end;
{$ENDREGION}

{$REGION 'Automation hooking Take 2'}
function DispParamToVariant(ParamType: TVarType;
  RawDispParam: Pointer; var DispParam: Variant): Pointer; forward;

type
  { Dispatch call descriptor }
  PCallDesc = ^TCallDesc;
  TCallDesc = packed record
    CallType: Byte;
    ArgCount: Byte;
    NamedArgCount: Byte;
    ArgTypes: array[0..255] of Byte;
  end;

procedure VarDispHandler2(Result: PVariant; const Instance: Variant;
  CallDesc: PCallDesc; Params: Pointer); cdecl;
var
  Msg: String;        //debug msg to output
  ParamPtr: Pointer;  //rolling pointer to next parameter
  NamePtr: PAnsiChar; //pointer to arg name
  NamedArgStart,      //arg position of 1st named argument (if any)
  ArgCount: Integer;  //argument counter
  Param: Variant;     //the parameter in a Variant
begin
  Msg := '';
  NamedArgStart := CallDesc.ArgCount - CallDesc.NamedArgCount;
  //After arg types, method name and named arg names are stored
  //Position pointer on method name
  NamePtr := @CallDesc.ArgTypes[CallDesc.ArgCount];
  ParamPtr := {$IFNDEF Delphi6AndAbove}@{$ENDIF}Params;
  //Loop thru parameters, concatenating them together as a string
  for ArgCount := 0 to Pred(CallDesc.ArgCount) do
  begin
    ParamPtr := DispParamToVariant(
      CallDesc.ArgTypes[ArgCount], ParamPtr, Param);
    //Inject named param name in string
    if ArgCount >= NamedArgStart then
    begin
      NamePtr := NamePtr + Succ(AnsiStrings.StrLen(NamePtr));
      Param := Format('%s := %s', [NamePtr, String(Param)]);
    end;
    if ArgCount < Pred(CallDesc.ArgCount) then
      Param := String(Param) + ' ';
    Msg := Msg + String(Param);
  end;
  DebugMsg(Msg, ShowPIDAndTIDInfo);
end;

//Takes pointer to one parameter from a list of disp params
//Sets Variant to match parameter and returns address of next parameter
function DispParamToVariant(ParamType: TVarType;
  RawDispParam: Pointer; var DispParam: Variant): Pointer;
var
  ParamPtrAdjust: Integer;
  VarFlag: Boolean;
const
  ParamPtrDefaultAdjust = SizeOf(Pointer);
  atTypeMask = $7F;
  atByRef    = $80;
  varWord64  = $0015;
{$IFNDEF Delphi6AndAbove}
  varInt64   = $0014;
{$ENDIF}
{$IFDEF Delphi4}
  varStrArg  = $0048;
{$ENDIF}
begin
  DispParam := Null;
  Result := RawDispParam;
  VarFlag := ParamType and atByRef > 0;
  //If parameter was passed by reference, must do extra de-reference
  if VarFlag then
    RawDispParam := Pointer(RawDispParam^);
  //Normally get to next parameter by adding 4 bytes,
  //as they either take up 4 bytes, or are represented by a pointer.
  //Parameters that are not passed by ref, but which
  //take up more than 4 bytes in the parameter list
  //have to have the rest of the occupied bytes skipped
  ParamPtrAdjust := ParamPtrDefaultAdjust;
  case ParamType and atTypeMask of
  {$IFDEF Delphi6AndAbove}
    varShortint, //N.B. shortints are actually passed through as varInteger
  {$ENDIF}
    varSmallint,
    varInteger: DispParam := Integer(RawDispParam^);
    varSingle: DispParam := Single(RawDispParam^);
    varDouble:
    begin
      DispParam := Double(RawDispParam^);
      if not VarFlag then
        //Swallow extra 4 bytes taken by this 8-byte Double
        Inc(ParamPtrAdjust, (SizeOf(Double) - ParamPtrDefaultAdjust) * Ord(not VarFlag));
    end;
    varCurrency:
    begin
      DispParam := Currency(RawDispParam^);
      if not VarFlag then
        //Swallow extra 4 bytes taken by this 8-byte Currency
        Inc(ParamPtrAdjust, SizeOf(Currency) - ParamPtrDefaultAdjust);
    end;
    varDate:
    begin
      DispParam := TDateTime(RawDispParam^);
      if not VarFlag then
        //Swallow extra 4 bytes taken by this 8-byte TDateTime
        Inc(ParamPtrAdjust, SizeOf(TDateTime) - ParamPtrDefaultAdjust);
    end;
    varOleStr: DispParam := WideString(PWideChar(RawDispParam));
    varBoolean: DispParam := WordBool(RawDispParam^);
    varVariant:
    begin
      DispParam := Variant(RawDispParam^);
      if not VarFlag then
        //Swallow extra 12 bytes taken by this 16-byte Variant
        Inc(ParamPtrAdjust, SizeOf(Variant) - ParamPtrDefaultAdjust);
    end;
  {$IFDEF Delphi6AndAbove}
    varLongWord, //N.B. long words are actually passed through as varInteger
    varWord,     //N.B. words are actually passed through as varInteger
  {$ENDIF}
    varByte: DispParam := LongWord(RawDispParam^);
  {$IFDEF Delphi6AndAbove} //Delphi 6 allows Int64's in Variants
    varInt64, //Int64's are actually passed as varWord64
    varWord64: DispParam := Int64(RawDispParam^);
  {$ENDIF}
    varStrArg: DispParam := String(PAnsiChar(RawDispParam^));
  {$IFDEF UNICODE}
    varUStrArg: DispParam := String(PChar(RawDispParam^)); //Unicode versions of Delphi
  {$ENDIF}
  else
    DispParam := Format('Parameter has unsupported TVarType value: %d', [ParamType])
  end;
  Inc(Integer(Result), ParamPtrAdjust);
end;
{$ENDREGION}

procedure RerouteAutomation;
begin
  {$IFDEF Delphi6AndAbove}
  TVarData(Msg).VType := varUnknown;
  {$ENDIF}
  //VarDispProc := @VarDispHandler1;
  VarDispProc := @VarDispHandler2;
end;

{$REGION 'Assertion hooking'}
procedure AssertErrorHandler(const Msg, Filename: string; LineNumber: Integer; ErrorAddr: Pointer);
begin
  //ErrorAddr is address Assert will return to
  //(i.e. instruction following call to Assert).
  //Subtract 1 from it to ensure it refers to part
  //of the Assert call, making it more appropriate
  DebugMsg('%s (%s, line %d, address $%x)',
    [Msg, Filename, LineNumber, Pred(Integer(ErrorAddr))],
    ShowPIDAndTIDInfo);
end;
{$ENDREGION}

procedure RerouteAssertions;
begin
  AssertErrorProc := @AssertErrorHandler;
end;

initialization
{$IFDEF HookAssertions}
  RerouteAssertions;
{$ENDIF}
{$IFDEF HookAutomation}
  RerouteAutomation;
{$ENDIF}
end.
